import { Controller, Post, Body, Req, Res, UseGuards } from '@nestjs/common';
import { Throttle } from '@nestjs/throttler';
import { Request, Response } from 'express';
import { AuthService } from './auth.service';
import { RegisterDto } from './dto/register.dto';
import { LoginDto } from './dto/login.dto';

@Controller('auth')
export class AuthController {
  constructor(private auth: AuthService) {}

  @Post('register')
  @Throttle({ default: { limit: 5, ttl: 60000 } })
  async register(@Body() body: RegisterDto, @Res({ passthrough: true }) res: Response) {
    const result = await this.auth.register(body);
    res.cookie(this.auth.getRefreshCookieName(), result.refreshToken, this.auth.getRefreshCookieOptions());
    return { user: result.user, accessToken: result.accessToken, expiresIn: result.expiresIn };
  }

  @Post('login')
  @Throttle({ default: { limit: 5, ttl: 60000 } })
  async login(@Body() body: LoginDto, @Res({ passthrough: true }) res: Response) {
    const result = await this.auth.login(body);
    res.cookie(this.auth.getRefreshCookieName(), result.refreshToken, this.auth.getRefreshCookieOptions());
    return { user: result.user, accessToken: result.accessToken, expiresIn: result.expiresIn };
  }

  @Post('refresh')
  async refresh(@Req() req: Request, @Res({ passthrough: true }) res: Response) {
    const token = req.cookies?.[this.auth.getRefreshCookieName()] ?? req.body?.refreshToken;
    const result = await this.auth.refresh(token);
    res.cookie(this.auth.getRefreshCookieName(), result.refreshToken, this.auth.getRefreshCookieOptions());
    return { user: result.user, accessToken: result.accessToken, expiresIn: result.expiresIn };
  }

  @Post('logout')
  async logout(@Req() req: Request, @Res({ passthrough: true }) res: Response) {
    const token = req.cookies?.[this.auth.getRefreshCookieName()];
    await this.auth.logout(token);
    res.clearCookie(this.auth.getRefreshCookieName(), { path: '/', httpOnly: true });
    return { ok: true };
  }
}
