import { Injectable, CanActivate, ExecutionContext, ForbiddenException } from '@nestjs/common';
import { PrismaService } from '../../prisma/prisma.service';
import { SubscriptionStatus } from '@prisma/client';

const ALLOWED_STATUSES: SubscriptionStatus[] = ['active', 'trialing'];

@Injectable()
export class SubscriptionGuard implements CanActivate {
  constructor(private prisma: PrismaService) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const req = context.switchToHttp().getRequest();
    const user = req.user;
    if (!user) return false;
    if (user.platformRole === 'super_admin') return true;
    const tenantId = req.tenantId ?? user.tenantId;
    if (!tenantId) return true;
    const sub = await this.prisma.subscription.findUnique({
      where: { tenantId },
    });
    if (!sub) throw new ForbiddenException('Abonelik bulunamadı');
    if (ALLOWED_STATUSES.includes(sub.status)) return true;
    const path = req.route?.path ?? req.url;
    const billingPaths = ['/billing', '/subscription', '/stripe', '/payments', '/invoices'];
    const isBilling = billingPaths.some((p) => path.includes(p));
    if (isBilling) return true;
    throw new ForbiddenException('Aktif abonelik gerekli. Lütfen ödeme sayfasından yenileyin.');
  }
}
