import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  UseGuards,
  Req,
} from '@nestjs/common';
import { TenantService } from './tenant.service';
import { JwtAuthGuard, TenantGuard, SubscriptionGuard, RolesGuard, Roles } from '../auth/guards';
import { CreateLocationDto, UpdateLocationDto } from './dto/location.dto';
import { CreateTableDto, UpdateTableDto } from './dto/table.dto';

@Controller('tenant')
@UseGuards(JwtAuthGuard, TenantGuard, SubscriptionGuard)
export class TenantController {
  constructor(private tenant: TenantService) {}

  @Get('me')
  async getMe(@Req() req: any) {
    const tenantId = req.tenantId ?? req.user?.tenantId;
    if (!tenantId) return { tenant: null };
    return this.tenant.getTenantForUser(req.user.id, tenantId);
  }

  @Get('locations')
  async listLocations(@Req() req: any) {
    return this.tenant.listLocations(req.tenantId);
  }

  @Post('locations')
  @Roles('owner', 'admin', 'editor')
  async createLocation(@Req() req: any, @Body() body: CreateLocationDto) {
    return this.tenant.createLocation(req.tenantId, req.user.id, body);
  }

  @Put('locations/:id')
  @Roles('owner', 'admin', 'editor')
  async updateLocation(
    @Req() req: any,
    @Param('id') id: string,
    @Body() body: UpdateLocationDto,
  ) {
    return this.tenant.updateLocation(req.tenantId, req.user.id, id, body);
  }

  @Delete('locations/:id')
  @Roles('owner', 'admin')
  async deleteLocation(@Req() req: any, @Param('id') id: string) {
    return this.tenant.deleteLocation(req.tenantId, req.user.id, id);
  }

  @Get('tables')
  async listTables(@Req() req: any) {
    const locationId = (req.query as { locationId?: string }).locationId;
    return this.tenant.listTables(req.tenantId, locationId);
  }

  @Post('tables')
  @Roles('owner', 'admin', 'editor')
  async createTable(@Req() req: any, @Body() body: CreateTableDto) {
    return this.tenant.createTable(req.tenantId, req.user.id, body);
  }

  @Put('tables/:id')
  @Roles('owner', 'admin', 'editor')
  async updateTable(
    @Req() req: any,
    @Param('id') id: string,
    @Body() body: UpdateTableDto,
  ) {
    return this.tenant.updateTable(req.tenantId, req.user.id, id, body);
  }

  @Delete('tables/:id')
  @Roles('owner', 'admin')
  async deleteTable(@Req() req: any, @Param('id') id: string) {
    return this.tenant.deleteTable(req.tenantId, req.user.id, id);
  }
}
