import { useEffect, useState } from 'react';
import {
  View,
  Text,
  TextInput,
  TouchableOpacity,
  StyleSheet,
  ScrollView,
  ActivityIndicator,
} from 'react-native';
import { productApi } from '../lib/api';

export function ProductEditScreen({ route, navigation }: any) {
  const productId = route.params?.productId;
  const [categories, setCategories] = useState<any[]>([]);
  const [form, setForm] = useState({
    name: '',
    description: '',
    price: '',
    currency: 'TRY',
    categoryId: '',
    isActive: true,
  });
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState('');

  useEffect(() => {
    productApi.listCategories().then(setCategories);
    if (productId) {
      setLoading(true);
      productApi
        .getProduct(productId)
        .then((p) => {
          setForm({
            name: p.name,
            description: p.description ?? '',
            price: String(p.price),
            currency: p.currency ?? 'TRY',
            categoryId: p.categoryId,
            isActive: p.isActive ?? true,
          });
        })
        .catch(() => setError('Ürün bulunamadı'))
        .finally(() => setLoading(false));
    } else {
      if (categories.length) setForm((f) => ({ ...f, categoryId: categories[0].id }));
    }
  }, [productId]);

  useEffect(() => {
    if (categories.length && !form.categoryId) setForm((f) => ({ ...f, categoryId: categories[0].id }));
  }, [categories]);

  const handleSave = async () => {
    setError('');
    setSaving(true);
    try {
      if (productId) {
        await productApi.updateProduct(productId, {
          ...form,
          price: parseFloat(form.price),
        });
      } else {
        await productApi.createProduct({
          ...form,
          price: parseFloat(form.price),
          categoryId: form.categoryId || categories[0]?.id,
        });
      }
      navigation.goBack();
    } catch (err: any) {
      setError(err.message || 'Kayıt başarısız');
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <View style={styles.centered}>
        <ActivityIndicator size="large" color="#059669" />
      </View>
    );
  }

  return (
    <ScrollView style={styles.container} contentContainerStyle={styles.content}>
      {error ? <Text style={styles.error}>{error}</Text> : null}
      <Text style={styles.label}>Ürün adı</Text>
      <TextInput
        style={styles.input}
        value={form.name}
        onChangeText={(t) => setForm((f) => ({ ...f, name: t }))}
        placeholder="Ad"
      />
      <Text style={styles.label}>Açıklama</Text>
      <TextInput
        style={[styles.input, styles.textArea]}
        value={form.description}
        onChangeText={(t) => setForm((f) => ({ ...f, description: t }))}
        placeholder="Açıklama"
        multiline
      />
      <Text style={styles.label}>Fiyat</Text>
      <TextInput
        style={styles.input}
        value={form.price}
        onChangeText={(t) => setForm((f) => ({ ...f, price: t }))}
        placeholder="0.00"
        keyboardType="decimal-pad"
      />
      <Text style={styles.label}>Kategori</Text>
      <View style={styles.categoryList}>
        {categories.map((c) => (
          <TouchableOpacity
            key={c.id}
            style={[styles.chip, form.categoryId === c.id && styles.chipActive]}
            onPress={() => setForm((f) => ({ ...f, categoryId: c.id }))}
          >
            <Text style={[styles.chipText, form.categoryId === c.id && styles.chipTextActive]}>
              {c.name}
            </Text>
          </TouchableOpacity>
        ))}
      </View>
      <TouchableOpacity
        style={[styles.button, saving && styles.buttonDisabled]}
        onPress={handleSave}
        disabled={saving}
      >
        {saving ? (
          <ActivityIndicator color="#fff" />
        ) : (
          <Text style={styles.buttonText}>{productId ? 'Güncelle' : 'Kaydet'}</Text>
        )}
      </TouchableOpacity>
    </ScrollView>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#f8fafc' },
  content: { padding: 16, paddingBottom: 40 },
  centered: { flex: 1, justifyContent: 'center', alignItems: 'center' },
  error: { color: '#dc2626', marginBottom: 12, fontSize: 14 },
  label: { fontSize: 14, fontWeight: '500', color: '#475569', marginBottom: 6 },
  input: {
    borderWidth: 1,
    borderColor: '#cbd5e1',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    marginBottom: 16,
    backgroundColor: '#fff',
  },
  textArea: { minHeight: 80 },
  categoryList: { flexDirection: 'row', flexWrap: 'wrap', gap: 8, marginBottom: 20 },
  chip: {
    paddingVertical: 8,
    paddingHorizontal: 14,
    borderRadius: 20,
    backgroundColor: '#e2e8f0',
  },
  chipActive: { backgroundColor: '#059669' },
  chipText: { fontSize: 14, color: '#475569' },
  chipTextActive: { color: '#fff' },
  button: {
    backgroundColor: '#059669',
    borderRadius: 8,
    padding: 14,
    alignItems: 'center',
  },
  buttonDisabled: { opacity: 0.7 },
  buttonText: { color: '#fff', fontSize: 16, fontWeight: '600' },
});
