import { useEffect, useState } from 'react';
import {
  View,
  Text,
  FlatList,
  TouchableOpacity,
  StyleSheet,
  ActivityIndicator,
} from 'react-native';
import { productApi, tenantApi } from '../lib/api';

export function ProductsScreen({ navigation }: any) {
  const [products, setProducts] = useState<any[]>([]);
  const [tenant, setTenant] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    tenantApi.getMe().then(setTenant).catch(() => {});
    productApi.listProducts().then(setProducts).finally(() => setLoading(false));
  }, []);

  const refresh = () => {
    setLoading(true);
    productApi.listProducts().then(setProducts).finally(() => setLoading(false));
  };

  if (loading && products.length === 0) {
    return (
      <View style={styles.centered}>
        <ActivityIndicator size="large" color="#059669" />
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <TouchableOpacity
        style={styles.fab}
        onPress={() => navigation.navigate('ProductEdit', { productId: null })}
      >
        <Text style={styles.fabText}>+ Ürün ekle</Text>
      </TouchableOpacity>
      <FlatList
        data={products}
        keyExtractor={(item) => item.id}
        refreshing={loading}
        onRefresh={refresh}
        renderItem={({ item }) => (
          <TouchableOpacity
            style={styles.card}
            onPress={() => navigation.navigate('ProductEdit', { productId: item.id })}
          >
            <View style={styles.cardContent}>
              <Text style={styles.productName}>{item.name}</Text>
              <Text style={styles.price}>
                {Number(item.price).toLocaleString('tr-TR')} {item.currency}
              </Text>
              {item.model3dUrl ? (
                <TouchableOpacity
                  style={styles.view3dBtn}
                  onPress={() =>
                    navigation.navigate('Viewer3D', { url: item.model3dUrl, name: item.name })
                  }
                >
                  <Text style={styles.view3dText}>3D Görünüm</Text>
                </TouchableOpacity>
              ) : null}
            </View>
          </TouchableOpacity>
        )}
      />
    </View>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#f8fafc', padding: 16 },
  centered: { flex: 1, justifyContent: 'center', alignItems: 'center' },
  card: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    marginBottom: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  cardContent: {},
  productName: { fontSize: 16, fontWeight: '600', color: '#1e293b' },
  price: { fontSize: 14, color: '#059669', marginTop: 4 },
  view3dBtn: { marginTop: 8 },
  view3dText: { fontSize: 13, color: '#059669', fontWeight: '500' },
  fab: {
    position: 'absolute',
    bottom: 24,
    right: 16,
    backgroundColor: '#059669',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 12,
    zIndex: 10,
  },
  fabText: { color: '#fff', fontWeight: '600', fontSize: 14 },
});
