'use client';

import { useEffect, useState } from 'react';
import { tenantApi } from '@/lib/api';

export default function LocationsPage() {
  const [locations, setLocations] = useState<any[]>([]);
  const [tables, setTables] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showNewLocation, setShowNewLocation] = useState(false);
  const [showNewTable, setShowNewTable] = useState<string | null>(null);
  const [locForm, setLocForm] = useState({ name: '', address: '', city: '' });
  const [tableForm, setTableForm] = useState({ name: '', locationId: '' });

  useEffect(() => {
    tenantApi.listLocations().then((locs) => {
      setLocations(locs);
      return tenantApi.listTables();
    }).then(setTables).finally(() => setLoading(false));
  }, []);

  const handleCreateLocation = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const loc = await tenantApi.createLocation(locForm);
      setLocations((l) => [...l, loc]);
      setLocForm({ name: '', address: '', city: '' });
      setShowNewLocation(false);
    } catch (err) {
      console.error(err);
    }
  };

  const handleCreateTable = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!tableForm.name.trim() || !tableForm.locationId) return;
    try {
      const t = await tenantApi.createTable({ name: tableForm.name.trim(), locationId: tableForm.locationId });
      setTables((prev) => [...prev, t]);
      setTableForm({ name: '', locationId: '' });
      setShowNewTable(null);
    } catch (err) {
      console.error(err);
    }
  };

  const tablesByLocation = locations.map((loc) => ({
    ...loc,
    tables: tables.filter((t) => t.locationId === loc.id),
  }));

  if (loading) return <div className="text-slate-500">Yükleniyor...</div>;

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold text-slate-800">Şubeler & Masalar</h1>
        {!showNewLocation && (
          <button
            onClick={() => setShowNewLocation(true)}
            className="px-4 py-2 bg-emerald-600 text-white rounded-lg text-sm font-medium hover:bg-emerald-700"
          >
            Yeni şube
          </button>
        )}
      </div>
      {showNewLocation && (
        <form onSubmit={handleCreateLocation} className="mb-6 p-4 bg-white rounded-xl border border-slate-200 space-y-2">
          <input
            type="text"
            placeholder="Şube adı"
            value={locForm.name}
            onChange={(e) => setLocForm((f) => ({ ...f, name: e.target.value }))}
            className="w-full px-4 py-2 border border-slate-300 rounded-lg"
            required
          />
          <input
            type="text"
            placeholder="Adres"
            value={locForm.address}
            onChange={(e) => setLocForm((f) => ({ ...f, address: e.target.value }))}
            className="w-full px-4 py-2 border border-slate-300 rounded-lg"
          />
          <input
            type="text"
            placeholder="Şehir"
            value={locForm.city}
            onChange={(e) => setLocForm((f) => ({ ...f, city: e.target.value }))}
            className="w-full px-4 py-2 border border-slate-300 rounded-lg"
          />
          <div className="flex gap-2">
            <button type="submit" className="px-4 py-2 bg-emerald-600 text-white rounded-lg">Ekle</button>
            <button type="button" onClick={() => setShowNewLocation(false)} className="px-4 py-2 border rounded-lg">İptal</button>
          </div>
        </form>
      )}
      <div className="space-y-6">
        {tablesByLocation.map((loc) => (
          <div key={loc.id} className="bg-white rounded-xl border border-slate-200 overflow-hidden">
            <div className="px-4 py-3 bg-slate-50 border-b border-slate-200 font-medium text-slate-800">
              {loc.name}
              {loc.address && <span className="text-slate-500 text-sm font-normal ml-2">— {loc.address}</span>}
            </div>
            <div className="p-4">
              <div className="flex flex-wrap gap-2 mb-2">
                {loc.tables.map((t: any) => (
                  <span
                    key={t.id}
                    className="px-3 py-1.5 bg-slate-100 rounded-lg text-sm text-slate-700"
                  >
                    {t.name}
                  </span>
                ))}
                {showNewTable === loc.id ? (
                  <form
                    onSubmit={(e) => {
                      e.preventDefault();
                      handleCreateTable(e);
                    }}
                    className="flex gap-2 items-center"
                    onClick={(ev) => ev.stopPropagation()}
                  >
                    <input
                      type="text"
                      placeholder="Masa adı"
                      value={tableForm.locationId === loc.id ? tableForm.name : ''}
                      onChange={(e) => setTableForm({ name: e.target.value, locationId: loc.id })}
                      className="px-3 py-1.5 border rounded-lg text-sm w-28"
                    />
                    <button type="submit" className="text-emerald-600 text-sm font-medium">Ekle</button>
                    <button type="button" onClick={() => setShowNewTable(null)} className="text-slate-500 text-sm">İptal</button>
                  </form>
                ) : (
                  <button
                    type="button"
                    onClick={() => { setShowNewTable(loc.id); setTableForm({ name: '', locationId: loc.id }); }}
                    className="px-3 py-1.5 border border-dashed border-slate-300 rounded-lg text-sm text-slate-500 hover:bg-slate-50"
                  >
                    + Masa ekle
                  </button>
                )}
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
