'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import { productApi } from '@/lib/api';

export default function ProductsPage() {
  const [products, setProducts] = useState<any[]>([]);
  const [categories, setCategories] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    Promise.all([productApi.listProducts(), productApi.listCategories()]).then(
      ([p, c]) => {
        setProducts(p);
        setCategories(c);
      },
    ).finally(() => setLoading(false));
  }, []);

  if (loading) return <div className="text-slate-500">Yükleniyor...</div>;

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold text-slate-800">Ürünler</h1>
        <Link
          href="/dashboard/products/new"
          className="px-4 py-2 bg-emerald-600 text-white rounded-lg text-sm font-medium hover:bg-emerald-700"
        >
          Yeni ürün
        </Link>
      </div>
      <div className="bg-white rounded-xl border border-slate-200 overflow-hidden">
        <table className="w-full">
          <thead className="bg-slate-50 border-b border-slate-200">
            <tr>
              <th className="text-left py-3 px-4 text-sm font-medium text-slate-600">Ürün</th>
              <th className="text-left py-3 px-4 text-sm font-medium text-slate-600">Kategori</th>
              <th className="text-left py-3 px-4 text-sm font-medium text-slate-600">Fiyat</th>
              <th className="text-left py-3 px-4 text-sm font-medium text-slate-600">Durum</th>
              <th className="w-20" />
            </tr>
          </thead>
          <tbody>
            {products.map((p) => (
              <tr key={p.id} className="border-b border-slate-100 hover:bg-slate-50/50">
                <td className="py-3 px-4">
                  <div className="font-medium text-slate-800">{p.name}</div>
                  {p.description && (
                    <div className="text-sm text-slate-500 truncate max-w-xs">{p.description}</div>
                  )}
                </td>
                <td className="py-3 px-4 text-slate-600">{p.category?.name ?? '—'}</td>
                <td className="py-3 px-4">
                  {Number(p.price).toLocaleString('tr-TR')} {p.currency}
                </td>
                <td className="py-3 px-4">
                  <span
                    className={`text-xs px-2 py-1 rounded-full ${
                      p.isActive ? 'bg-emerald-100 text-emerald-700' : 'bg-slate-100 text-slate-600'
                    }`}
                  >
                    {p.isActive ? 'Aktif' : 'Pasif'}
                  </span>
                </td>
                <td className="py-3 px-4">
                  <Link
                    href={`/dashboard/products/${p.id}`}
                    className="text-emerald-600 hover:underline text-sm"
                  >
                    Düzenle
                  </Link>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
        {products.length === 0 && (
          <div className="py-12 text-center text-slate-500">
            Henüz ürün yok. <Link href="/dashboard/products/new" className="text-emerald-600 hover:underline">İlk ürünü ekleyin</Link>.
          </div>
        )}
      </div>
    </div>
  );
}
