'use client';

import { useEffect, useState } from 'react';
import { tenantApi } from '@/lib/api';

const API_URL = process.env.NEXT_PUBLIC_API_URL || 'http://localhost:4000';

async function fetchQrBlobUrl(path: string, format: 'png' | 'svg'): Promise<string> {
  const token = localStorage.getItem('accessToken');
  const res = await fetch(`${API_URL}${path}?format=${format}`, {
    headers: token ? { Authorization: `Bearer ${token}` } : {},
    credentials: 'include',
  });
  if (!res.ok) throw new Error('QR alınamadı');
  const blob = await res.blob();
  return URL.createObjectURL(blob);
}

export default function QrPage() {
  const [tenant, setTenant] = useState<any>(null);
  const [locations, setLocations] = useState<any[]>([]);
  const [tables, setTables] = useState<any[]>([]);
  const [format, setFormat] = useState<'png' | 'svg'>('png');
  const [qrUrls, setQrUrls] = useState<Record<string, string>>({});

  useEffect(() => {
    tenantApi.getMe().then((t) => setTenant(t));
    tenantApi.listLocations().then(setLocations);
    tenantApi.listTables().then(setTables);
  }, []);

  useEffect(() => {
    if (!tenant) return;
    const keys = ['tenant', ...locations.map((l) => `loc-${l.id}`), ...tables.map((t) => `table-${t.id}`)];
    keys.forEach((key) => {
      const path =
        key === 'tenant'
          ? '/tenant/qr/tenant'
          : key.startsWith('loc-')
            ? `/tenant/qr/location/${key.replace('loc-', '')}`
            : `/tenant/qr/table/${key.replace('table-', '')}`;
      fetchQrBlobUrl(path, format).then((url) => setQrUrls((u) => ({ ...u, [key]: url })));
    });
    return () => {
      Object.values(qrUrls).forEach(URL.revokeObjectURL);
    };
  }, [tenant?.id, locations.length, tables.length, format]);

  if (!tenant) return <div className="text-slate-500">Yükleniyor...</div>;

  const menuBase = typeof window !== 'undefined'
    ? window.location.origin.replace('app.', 'menu.')
    : 'https://menu.domain.com';

  return (
    <div>
      <h1 className="text-2xl font-bold text-slate-800 mb-6">QR Kodları</h1>
      <div className="mb-4 flex items-center gap-2">
        <span className="text-sm text-slate-600">Format:</span>
        <button
          type="button"
          onClick={() => setFormat('png')}
          className={`px-3 py-1 rounded text-sm ${format === 'png' ? 'bg-emerald-600 text-white' : 'bg-slate-200'}`}
        >
          PNG
        </button>
        <button
          type="button"
          onClick={() => setFormat('svg')}
          className={`px-3 py-1 rounded text-sm ${format === 'svg' ? 'bg-emerald-600 text-white' : 'bg-slate-200'}`}
        >
          SVG
        </button>
      </div>
      <div className="grid gap-6">
        <div className="bg-white rounded-xl border border-slate-200 p-4">
          <h2 className="font-semibold text-slate-800 mb-2">Genel menü (tüm restoran)</h2>
          <p className="text-sm text-slate-500 mb-2">Link: {menuBase}/r/{tenant.slug}</p>
          <div className="flex items-start gap-4">
            {qrUrls.tenant && (
              <>
                <img
                  src={qrUrls.tenant}
                  alt="QR"
                  className="w-40 h-40 border border-slate-200 rounded-lg"
                  style={{ objectFit: 'contain' }}
                />
                <a
                  href={qrUrls.tenant}
                  download={`qr-${tenant.slug}.${format}`}
                  className="text-sm text-emerald-600 hover:underline"
                >
                  İndir
                </a>
              </>
            )}
          </div>
        </div>
        {locations.map((loc) => (
          <div key={loc.id} className="bg-white rounded-xl border border-slate-200 p-4">
            <h2 className="font-semibold text-slate-800 mb-2">Şube: {loc.name}</h2>
            <p className="text-sm text-slate-500 mb-2">Link: {menuBase}/r/{tenant.slug}/l/{loc.id}</p>
            <div className="flex items-start gap-4">
              {qrUrls[`loc-${loc.id}`] && (
                <>
                  <img
                    src={qrUrls[`loc-${loc.id}`]}
                    alt="QR"
                    className="w-40 h-40 border border-slate-200 rounded-lg"
                    style={{ objectFit: 'contain' }}
                  />
                  <a
                    href={qrUrls[`loc-${loc.id}`]}
                    download={`qr-${loc.name}.${format}`}
                    className="text-sm text-emerald-600 hover:underline"
                  >
                    İndir
                  </a>
                </>
              )}
            </div>
            <div className="mt-4 pl-4 border-l-2 border-slate-100">
              <p className="text-sm font-medium text-slate-600 mb-2">Masalar</p>
              <div className="flex flex-wrap gap-4">
                {tables.filter((t) => t.locationId === loc.id).map((t) => (
                  <div key={t.id} className="flex flex-col items-center">
                    {qrUrls[`table-${t.id}`] && (
                      <>
                        <img
                          src={qrUrls[`table-${t.id}`]}
                          alt={t.name}
                          className="w-28 h-28 border border-slate-200 rounded-lg"
                          style={{ objectFit: 'contain' }}
                        />
                        <span className="text-xs text-slate-600 mt-1">{t.name}</span>
                        <a
                          href={qrUrls[`table-${t.id}`]}
                          download={`qr-${t.name}.${format}`}
                          className="text-xs text-emerald-600 hover:underline"
                        >
                          İndir
                        </a>
                      </>
                    )}
                  </div>
                ))}
              </div>
            </div>
          </div>
        ))}
      </div>
      <p className="mt-4 text-sm text-slate-500">
        QR kodları oturum açıkken API üzerinden alınır.
      </p>
    </div>
  );
}
